//---------------------------------------------------------------------------
// see: Application Note 39 - Calculating Great Circle Distances
//      <www.micromegacorp.com/appnotes.html>
//
// This program demonstrates the calculation of great circle distances.
// It reads the current latitude and longitude position from a GPS receiver,
// then calculates the great circle distance to a number of stored locations.
// The program can also be run without a GPS connection by selecting one of
// the pre-defined locations as the 'from' location.
//
// Note: To use this example, the uM-FPU V3.1 chip must be programmed with
// the user-defined functions contained in the file GCdistance.fpu located in
// the hardware/libraries/Fpu/Examples/GCdistance subdirectory of the Arduino
// application directory.
// @author Cam Thompson, Micromega Corporation
// <www.micromegacorp.com>
//
// @version
//  October 2, 2011
//  - updated for new SPI and FpuSerial libraries
//  December 15, 2008
//  - original version
//---------------------------------------------------------------------------

#include <SPI.h>
#include <Fpu.h>
#include <FpuSerial.h>

//------------------ uM-FPU register definitions ----------------------------
#define lat1            10          // uM-FPU register 10
#define long1           11          // uM-FPU register 11
#define lat2            12          // uM-FPU register 12
#define long2           13          // uM-FPU register 13

#define dist_nm         20          // distance - nautical miles
#define dist_km         21          // distance - kilometers
#define dist_miles      22          // distance - miles

  //-------------------- uM-FPU Function Definitions ------------------------
#define getID           0           // uM-FPU user function 0
#define getDistance     1           // uM-FPU user function 1
#define getLocation     2           // uM-FPU user function 2
#define getLatLong      3           // uM-FPU user function 3
#define radiansToDM     4           // uM-FPU user function 4
#define readNMEA        5           // uM-FPU user function 5
#define parseGPRMC      6           // uM-FPU user function 6
#define NMEA_Degrees    7           // uM-FPU user function 7

//-------------------- setup ------------------------------------------------

void setup()
{
  Serial.begin(9600);
  Serial.println("Great Circle Distance");
  
  SPI.begin();
  Fpu.begin();
  
  if (Fpu.sync() == SYNC_CHAR)
    FpuSerial.printVersionln();
  else
  {
    Serial.println("FPU not detected");
    while(1) ; // stop if FPU not detected
  }  
}

//-------------------- loop -------------------------------------------------

void loop()
{
  byte from;
  
  // check that the user-defined functions are loaded
  Fpu.write(CLR0, FCALL, getID);
  Fpu.write(SELECTA, 0);
  Fpu.wait();
  Fpu.write(LREADBYTE);
  if (Fpu.read() != 39)
  {
    Serial.println("GCdistance.fpu functions not loaded.");
    while(true) ;
  }

  // if from = 99, read current location from GPS
  // if from <> 99, use a stored location (0 to 7)
  from = 0;

  Serial.print("\r\nFrom:\r\n  ");
  if (from == 99)
  {
    Serial.print("Reading GPS ...");
    Fpu.write(FCALL, readNMEA);
    Fpu.wait();

    Fpu.write(STRSET);
    Fpu.writeString("Location");
  }
  else
  {
    Fpu.write(LONGBYTE, from, FCALL, getLocation);
    Fpu.write(COPY, lat2, lat1);
    Fpu.write(COPY, long2, long1);
  }

  // display latitude and longitude
  Fpu.write(STRINS);
  Fpu.writeString(" (");
  Fpu.write(SELECTA, lat1);
  Fpu.write(FCALL, radiansToDM, STRINS);
  Fpu.writeString(", ");
  Fpu.write(SELECTA, long1);
  Fpu.write(FCALL, radiansToDM, STRINS);
  Fpu.writeString(")");
  FpuSerial.printString();

  // display distance to all other stored locations
  Serial.print("\r\n\r\nTo:");

  for(int index = 0; index < 8; index++)
  {
    if (from != index)
    {
      Serial.print("\r\n  ");
      Fpu.write(LONGBYTE, index);
      Fpu.write(FCALL, getLocation, STRINS);
      Fpu.writeString(" (");
      Fpu.write(SELECTA, lat2);
      Fpu.write(FCALL, radiansToDM, STRINS);
      Fpu.writeString(", ");
      Fpu.write(SELECTA, long2);
      Fpu.write(FCALL, radiansToDM, STRINS);
      Fpu.writeString(")");
      FpuSerial.printString();

      Serial.print("\r\n   ");
      Fpu.write(SELECTA, dist_nm);
      Fpu.write(FCALL, getDistance, FSET0);
      FpuSerial.printFloat(92);
      Serial.print(" nautical miles\r\n   ");

      Fpu.write(SELECTA, dist_km, FSET, dist_nm);
      Fpu.write(FWRITE0);
      Fpu.writeFloat(1.852);
      Fpu.write(FMUL0);
      Fpu.write(SELECTA, dist_miles, FSET, dist_km);
      Fpu.write(FCNV, 13);
      FpuSerial.printFloat(92);
      Serial.print(" miles\r\n   ");
		  	  
      Fpu.write(SELECTA, dist_km);
      FpuSerial.printFloat(92);
      Serial.println(" kilometers");
    }
  }

  Serial.println("\r\nDone.");
  while(1) ;
}
